function GHGM3_save_results(EMISSIONS, EMISSION_SOURCE, LURNZDATA, c, GISheader, pixel_size, output_path, last_year, GHGM_PDFS, GHGM_ASCI, ALL_YEARS )
%
% Land Use Intensity Module - function 3 (LUIM3)
%
% This file saves the maps of land use intensity in pdf and asc form.
% For dairy farms this is Milk solids per hectare.
% For sheep/beef farms this is stock units of sheep and beef cattle
%

% Code by Simon Anastasiadis : 2012-04-05

%% Parameters

% base year
base_year = 2008;

% module output folder
module_folder = '\Results 4 Greenhouse Gas Module';
% GIS output folder
GIS_folder    = '\Results 4 Greenhouse Gas Module\GIS data';
% PDf map output folder
pdfmap_folder = '\Results 4 Greenhouse Gas Module\LandUse Maps';

%% Set folder to store results

% make module folder
if ~exist([output_path,module_folder],'dir')
    mkdir([output_path,module_folder]);
end

% make GIS output folder
if ~exist([output_path,GIS_folder],'dir')
    mkdir([output_path,GIS_folder]);
end

% make PDF map folder
if ~exist([output_path,pdfmap_folder],'dir')
    mkdir([output_path,pdfmap_folder]);
end

%% Identify years of interest

% construct vector containing years of interest
if ALL_YEARS
    % we want all years
    years = (base_year:last_year).';
else
    % we want just the final year
    years = last_year;
end

%% Save ASCI maps
% This code saves asci maps of GHG emissions for the last simulation year.
% Producing maps of the other years requires modification of the intensity
% module to produce multiple years of intensity measures (see a similar
% location in the intensity module for instructions) followed by
% modifications to this module to loop the GHG calculation over multiple
% years.

% if saving ASCI maps
if GHGM_ASCI
    % loop through years
    for ii = 1:length(years)
        % if saving all years // always saving final year
        if ALL_YEARS || ii == length(years)
    
            % save GHG emissions as asci map
            MapData = EMISSIONS.total(:,[1, ii+1]);
            file_title = ['GHG_emissions(kg co2equiv per ha)_',num2str(years(ii)),'.asc'];
            SaveAsciMaps(MapData, GISheader, output_path, GIS_folder, file_title);
        end
    end

    % Remove MapData to save space
    clear('MapData','file_title')
end
 
%% Save PDF maps
% This code saves PDF maps of GHG emissions for the last simulation year.
% Producing maps of the other years requires modification of the intensity
% module to produce multiple years of intensity measures (see a similar
% location in the intensity module for instructions) followed by
% modifications to this module to loop the GHG calculation over multiple
% years.

% if saving pdf maps
if GHGM_PDFS
    % loop through years
    for ii = 1:length(years)
        % if saving all years // always saving final year
        if ALL_YEARS || ii == length(years)
    
            % save Dairy GHG emissions as PDF map
            MapData = EMISSIONS.dairy(:,[1, ii+1]);
            file_title = ['GHG Dairy emissions (kg co2-equiv per ha) ',num2str(years(ii))];
            SavePDFMaps(MapData, GISheader, output_path, pdfmap_folder, file_title);

            % save Sheep/Beef GHG emissions as PDF map
            MapData = EMISSIONS.sheep(:,[1, ii+1]);
            file_title = ['GHG Sheep Beef emissions (kg co2-equiv per ha) ',num2str(years(ii))];
            SavePDFMaps(MapData, GISheader, output_path, pdfmap_folder, file_title);

            % save forestry sequestration as PDF map
            MapData = EMISSIONS.forest(:,[1, ii+1]);
            % sequestration only
            iEmis = MapData(:,2) > 0;
            MapData(iEmis,2) = 0;
            MapData(:,2) = -MapData(:,2);
            % save maps
            file_title = ['GHG Forestry sequestration (kg co2-equiv per ha) ',num2str(years(ii))];
            SavePDFMaps(MapData, GISheader, output_path, pdfmap_folder, file_title);
            
            % save scrub sequestration as PDF map
            MapData = EMISSIONS.scrub(:,[1, ii+1]);
            % sequestration only
            iEmis = MapData(:,2) > 0;
            MapData(iEmis,2) = 0;
            MapData(:,2) = -MapData(:,2);
            % save maps
            file_title = ['GHG Scrub sequestration (kg co2-equiv per ha) ',num2str(years(ii))];
            SavePDFMaps(MapData, GISheader, output_path, pdfmap_folder, file_title);
        end
    end
    
    % Remove MapData to save space
    clear('MapData','file_title')
end

%% Save summary tables by Territorial Authority & Regional Council
% This code saves TA & RC summary tables for the last simulation year,
% as this is anticipated to be the year of greatest interest.

% save summary tables
SaveTASummaryTables(LURNZDATA(:,c.ta), EMISSIONS, pixel_size, output_path, module_folder, years(end) );
SaveRCSummaryTables(LURNZDATA(:,c.rc), EMISSIONS, pixel_size, output_path, module_folder, years(end) );

%% Save Vector of emissions over time

if ALL_YEARS
    
    % empty vector for results
    Total_emissions = zeros(size(years));

    % loop through years, summing emissions
    for ii = 1:length(years)

        Total_emissions(ii) = sum(EMISSIONS.total(:,ii+1));
    end

    % save summary table
    row_headers = {'years' ; 'emissions'};
    file = [output_path, module_folder, '\emissions path ',num2str(years(end)),'.xls'];

    xlswrite(file,row_headers,'Sheet1','A1');
    xlswrite(file,years','Sheet1','B1');
    xlswrite(file,Total_emissions','Sheet1','B2');
end

%%  Save Matlab output

% clear excess variables
clear('GIS_folder',  'pdfmap_folder' ); 

% save output
save([output_path,module_folder,'\GHGM_output_data'])

%% Inform User saving land use change is complete

msg = sprintf(' - - GHGM3 : save data complete');
disp(msg)

end

%% Subfunction : Save ASCI maps

function SaveAsciMaps(MapData, GISheader, output_path, GIS_folder, file_title)

% unpack GISheader
% Map_GISHeader = [Longitude(ncols);Latitude(nrows);Xll;Yll;CellSize;No_Value]
Map_Longitude = GISheader(1);
Map_Latitude  = GISheader(2);
Map_No_Value  = GISheader(6);

% construct empty map
Map = Map_No_Value * ones(Map_Latitude, Map_Longitude);

% fill map
Map(MapData(:,1)) = MapData(:,2);

% Save
file = [output_path,GIS_folder,'\',file_title];
SaveAsciiRaster(file, Map, GISheader);

end

%% Subfunction : Save Land Use Map as PDFs

function SavePDFMaps(MapData, GISheader, output_path, pdfmap_folder, file_title)

% Construct empty map grids
Map_Longitude = GISheader(1);
Map_Latitude  = GISheader(2);
mapgrid = nan(Map_Latitude, Map_Longitude);

% fill map grids
mapgrid(MapData(:,1)) = MapData(:,2);

% the way pcolor handles axis requires we flip the map before plotting
mapgrid = flipud(mapgrid);

% Special mapping commands
scrsz = get(0,'ScreenSize');
set(gcf,'Position',[30 40 scrsz(3)/2 scrsz(4)-140]);
pcolor(mapgrid); shading flat; axis equal tight off;

% add color bar / key
colorbar('location','southoutside');

% determine scale of shading
max_value = max(MapData(:,2));
caxis([0 ceil(max_value)]);

colordata = colormap;
colordata(1,:) = 0.66;
colormap(colordata);

% give title
title(file_title);

% save to pdf
save2pdf([output_path,pdfmap_folder,'\',file_title]);
close;

end

%% Subfunction : Save Territorial Authority Summary Tables

function SaveTASummaryTables(ta_data, EMISSIONS, pixel_size, output_path, module_folder, last_year)

% identify TAs
tas = unique(ta_data);
tas = tas(~isnan(tas));
tas = sort(tas);

%% Determine areas

% sum pixels
tas_values = zeros(length(tas), 4);

% fill tas_areas
for ii = 1:length(tas)
    % indicator of TA values
    iTA = ta_data==tas(ii);
    
    % Dairy
    tas_values(ii,1) = sum(EMISSIONS.dairy(iTA,end));
    % Sheep/beef
    tas_values(ii,2) = sum(EMISSIONS.sheep(iTA,end));
    % Forest
    tas_values(ii,3) = sum(EMISSIONS.forest(iTA,end));
    % scrub
    tas_values(ii,4) = sum(EMISSIONS.scrub(iTA,end));
    % total
    tas_values(ii,5) = tas_values(ii,1) + tas_values(ii,2) + tas_values(ii,3) + tas_values(ii,4);
        
end

% scale by pixel size
tas_values = tas_values * pixel_size;

%% Save

% labels
row_headers = {'TA code','TA';1,'Far North District';2,'Whangarei District';3,'Kaipara District';4,'Rodney District';5,'North Shore City';6,'Waitakere City';7,'Auckland City';8,'Manukau City';9,'Papakura District';10,'Franklin District';11,'Thames-Coromandel District';12,'Hauraki District';13,'Waikato District';15,'Matamata-Piako District';16,'Hamilton City';17,'Waipa District';18,'Otorohanga District';19,'South Waikato District';20,'Waitomo District';21,'Taupo District';22,'Western Bay of Plenty District';23,'Tauranga District';24,'Rotorua District';25,'Kawerau District';26,'Whakatane District';27,'Opotiki District';28,'Gisborne District';29,'Wairoa District';30,'Napier City';31,'Hastings District';32,'Central Hawkes Bay District';33,'New Plymouth District';34,'Stratford District';35,'South Taranaki District';36,'Ruapehu District';37,'Wanganui District';38,'Rangitikei District';39,'Manawatu District';40,'Palmerston North City';41,'Tararua District';42,'Horowhenua District';43,'Kapiti Coast';44,'Porirua City';45,'Hutt District';46,'Lower Hutt City';47,'Wellington City';48,'Masterton District';49,'Carterton District';50,'South Wairarapa District';51,'Tasman District';52,'Nelson City';53,'Marlborough District';54,'Kaikoura District';55,'Buller District';56,'Grey District';57,'Westland District';58,'Hurunui District';59,'Waimakariri District';60,'Christchurch City';61,'Banks Peninsula District';62,'Selwyn District';63,'Ashburton District';64,'Timaru District';65,'Mackenzie District';66,'Waimate District';68,'Waitaki District';69,'Central Otago District';70,'Queenstown-Lakes District';71,'Dunedin City';72,'Clutha District';73,'Southland District';74,'Gore District';75,'Invercargill City';999,'Area not in a TA';};
col_headers = {'Dairy emissions','Sheep/beef emissions','Forestry Net emissions','Scrub net emissions','Total emissions';'Kgs Co2-equivalents', [ ], [ ], [ ], [ ];};

file = [output_path,module_folder,'\TA GHG emissions table ',num2str(last_year),'.xls'];

xlswrite(file,last_year,'Sheet1','A1');
xlswrite(file,row_headers,'Sheet1','A2');
xlswrite(file,col_headers,'Sheet1','C1');
xlswrite(file,tas_values(1:end-1,:),'Sheet1','C3');

end

%% Subfunction : Save Regional Council Summary Tables

function SaveRCSummaryTables(rc_data, EMISSIONS, pixel_size, output_path, module_folder, last_year)

% identify RCs
rcs = unique(rc_data);
rcs = rcs(~isnan(rcs));
rcs = sort(rcs);

%% Determine areas

% count pixels
rcs_values = zeros(length(rcs), 5);

% fill rcs_areas
for ii = 1:length(rcs)
    % indicator of RC values
    iRC = rc_data==rcs(ii);
    
    % Dairy
    rcs_values(ii,1) = sum(EMISSIONS.dairy(iRC,end));
    % Sheep/beef
    rcs_values(ii,2) = sum(EMISSIONS.sheep(iRC,end));
    % Forest
    rcs_values(ii,3) = sum(EMISSIONS.forest(iRC,end));
    % scrub
    rcs_values(ii,4) = sum(EMISSIONS.scrub(iRC,end));
    % total
    rcs_values(ii,5) = rcs_values(ii,1) + rcs_values(ii,2) + rcs_values(ii,3) + rcs_values(ii,4);
    
end

% scale by pixel size
rcs_values = rcs_values * pixel_size;

%% Save

% labels
row_headers = {'RC code','RC';1,'Northland';2,'Auckland';3,'Waikato';4,'Bay of Plenty';5,'Gisborne';6,'Hawkes Bay';7,'Taranaki';8,'Manawatu-Wanganui';9,'Wellington';12,'West Coast';13,'Canterbury';14,'Otago';15,'Southland';16,'Tasman';17,'Nelson';18,'Marlborough';};
col_headers = {'Dairy emissions','Sheep/beef emissions','Net forestry emissions','Net scrub emissions','Total emissions';'Kgs Co2-equivalents', [ ], [ ], [ ], [ ];};

file = [output_path,module_folder,'\RC GHG emissions table ',num2str(last_year),'.xls'];

xlswrite(file,last_year,'Sheet1','A1');
xlswrite(file,row_headers,'Sheet1','A2');
xlswrite(file,col_headers,'Sheet1','C1');
xlswrite(file,rcs_values,'Sheet1','C3');

end

